let token = null;
let username = null;

async function request(url, method = 'GET', body = null) {
  const headers = {};
  if (token) headers['Authorization'] = 'Bearer ' + token;

  let options = { method, headers };
  if (body) {
    if (body instanceof FormData) {
      options.body = body;
    } else {
      headers['Content-Type'] = 'application/json';
      options.body = JSON.stringify(body);
    }
  }
  const res = await fetch(url, options);
  return res.json();
}

function showError(msg) {
  alert(msg);
}

async function register() {
  const username = document.getElementById('register-username').value.trim();
  const email = document.getElementById('register-email').value.trim();
  const password = document.getElementById('register-password').value;

  if (!username || !email || !password) {
    showError('Visi lauki ir obligāti');
    return;
  }

  const res = await request('/api/auth/register', 'POST', { username, email, password });
  if (res.error) {
    showError(res.error);
  } else {
    alert('Reģistrācija veiksmīga! Lūdzu pieslēdzieties.');
  }
}

async function login() {
  const email = document.getElementById('login-email').value.trim();
  const password = document.getElementById('login-password').value;

  if (!email || !password) {
    showError('Visi lauki ir obligāti');
    return;
  }

  const res = await request('/api/auth/login', 'POST', { email, password });
  if (res.error) {
    showError(res.error);
  } else {
    token = res.token;
    username = res.username;
    document.getElementById('auth-section').classList.add('hidden');
    document.getElementById('app').classList.remove('hidden');
    loadCategories();
    loadTasks();
    loadTrash();
    loadSharedTasks();
  }
}

function logout() {
  token = null;
  username = null;
  document.getElementById('auth-section').classList.remove('hidden');
  document.getElementById('app').classList.add('hidden');
}

async function loadCategories() {
  const categories = await request('/api/tasks/categories');
  const select = document.getElementById('category-select');
  select.innerHTML = '';
  categories.forEach(cat => {
    const opt = document.createElement('option');
    opt.value = cat.id;
    opt.textContent = cat.name;
    select.appendChild(opt);
  });
}

async function addCategory() {
  const name = document.getElementById('new-category').value.trim();
  if (!name) return alert('Ievadiet kategorijas nosaukumu');
  const res = await request('/api/tasks/categories', 'POST', { name });
  if (res.error) {
    showError(res.error);
  } else {
    loadCategories();
    document.getElementById('new-category').value = '';
  }
}

async function loadTasks() {
  const tasks = await request('/api/tasks');
  const container = document.getElementById('tasks-list');
  container.innerHTML = '';
  tasks.forEach(t => {
    const div = document.createElement('div');
    div.className = 'task';
    div.innerHTML = `
      <strong>ID: ${t.id}</strong> | <em>${t.category_name}</em><br>
      <b>${t.title}</b><br>
      <p>${t.description}</p>
      <button onclick="deleteTask(${t.id})">Dzēst</button>
    `;
    container.appendChild(div);
  });
}

async function addTask() {
  const title = document.getElementById('task-title').value.trim();
  const description = document.getElementById('task-desc').value.trim();
  const category_id = document.getElementById('category-select').value;
  const fileInput = document.getElementById('task-file');

  if (!title || !category_id) {
    showError('Nosaukums un kategorija obligāti');
    return;
  }

  const formData = new FormData();
  formData.append('title', title);
  formData.append('description', description);
  formData.append('category_id', category_id);
  if (fileInput.files.length > 0) {
    formData.append('file', fileInput.files[0]);
  }

  const res = await request('/api/tasks', 'POST', formData);
  if (res.error) {
    showError(res.error);
  } else {
    document.getElementById('task-title').value = '';
    document.getElementById('task-desc').value = '';
    fileInput.value = '';
    loadTasks();
  }
}

async function deleteTask(id) {
  if (!confirm('Vai tiešām vēlaties dzēst šo uzdevumu?')) return;
  const res = await request(`/api/tasks/${id}`, 'DELETE');
  if (res.error) {
    showError(res.error);
  } else {
    loadTasks();
    loadTrash();
  }
}

async function loadTrash() {
  const tasks = await request('/api/tasks/trash');
  const container = document.getElementById('trash-list');
  container.innerHTML = '';
  tasks.forEach(t => {
    const div = document.createElement('div');
    div.className = 'task';
    div.innerHTML = `
      <strong>ID: ${t.id}</strong> | <b>${t.title}</b><br>
      <p>${t.description}</p>
    `;
    container.appendChild(div);
  });
}

async function shareTask() {
  const task_id = Number(document.getElementById('share-task-id').value);
  const shared_with_username = document.getElementById('share-username').value.trim();

  if (!task_id || !shared_with_username) {
    showError('Ievadiet uzdevuma ID un lietotājvārdu');
    return;
  }

  const res = await request('/api/share', 'POST', { task_id, shared_with_username });
  if (res.error) {
    showError(res.error);
  } else {
    alert(res.message);
    document.getElementById('share-task-id').value = '';
    document.getElementById('share-username').value = '';
  }
}

async function loadSharedTasks() {
  const tasks = await request('/api/share/shared-with-me');
  const container = document.getElementById('shared-tasks-list');
  container.innerHTML = '';
  tasks.forEach(t => {
    const div = document.createElement('div');
    div.className = 'shared-task';
    div.innerHTML = `
      <strong>ID: ${t.id}</strong> | <em>${t.category_name}</em> | No: ${t.owner_username}<br>
      <b>${t.title}</b><br>
      <p>${t.description}</p>
    `;
    container.appendChild(div);
  });
}
